<?php
/**
 * WPG Documentation View — Comprehensive Plugin Guide
 *
 * @package aos-wp-governance
 * @since   1.0.0
 */
defined('ABSPATH') || exit;
?>

<div class="wrap wpg-wrap">
    <h1 class="wpg-title">
        <span class="dashicons dashicons-book"></span>
        <?php esc_html_e('Documentation', 'aos-wp-governance'); ?>
        <span class="wpg-version">v<?php echo esc_html(WPG_VERSION); ?></span>
    </h1>

    <p class="wpg-subtitle">
        <?php esc_html_e('Everything you need to know about governing AI agents on your WordPress site.', 'aos-wp-governance'); ?>
    </p>

    <!-- Tab Navigation -->
    <nav class="wpg-docs-tabs" id="wpg-docs-tabs">
        <button class="wpg-docs-tab active" data-tab="getting-started">🚀 Getting Started</button>
        <button class="wpg-docs-tab" data-tab="how-it-works">⚙️ How It Works</button>
        <button class="wpg-docs-tab" data-tab="writing-policies">📝 Writing Policies</button>
        <button class="wpg-docs-tab" data-tab="conditions">🔧 Conditions Reference</button>
        <button class="wpg-docs-tab" data-tab="builtin-policies">🛡️ Built-in Policies</button>
        <button class="wpg-docs-tab" data-tab="abilities">📋 Abilities Reference</button>
        <button class="wpg-docs-tab" data-tab="rest-api">🔌 REST API</button>
        <button class="wpg-docs-tab" data-tab="troubleshooting">🔍 Troubleshooting</button>
        <button class="wpg-docs-tab" data-tab="faq">❓ FAQ</button>
    </nav>

    <!-- ==================== GETTING STARTED ==================== -->
    <div class="wpg-docs-panel active" id="tab-getting-started">
        <div class="wpg-card">
            <h2>🚀 Getting Started with WP Governance</h2>

            <div class="wpg-docs-intro">
                <p><strong>AOS WP Governance</strong> is a constitutional governance layer for WordPress that controls what AI agents (Claude, ChatGPT, Cursor, Copilot, etc.) can and cannot do on your site when they connect through the <strong>Model Context Protocol (MCP)</strong>.</p>
                <p>Think of it as a <strong>security guard for AI</strong> — every action an AI wants to take is checked against your policies before it's allowed to proceed.</p>
            </div>

            <h3>Quick Setup (3 Steps)</h3>
            <div class="wpg-docs-steps">
                <div class="wpg-docs-step">
                    <div class="wpg-step-number">1</div>
                    <div>
                        <h4>Activate the Plugin</h4>
                        <p>Once activated, 6 built-in security policies are automatically installed. These protect against the most common AI risks: unauthorized admin creation, mass content deletion, plugin installation, file editing, and settings tampering.</p>
                        <p class="wpg-docs-result">✅ <strong>Your site is already protected.</strong> No configuration needed for basic safety.</p>
                    </div>
                </div>
                <div class="wpg-docs-step">
                    <div class="wpg-step-number">2</div>
                    <div>
                        <h4>Review Your Policies</h4>
                        <p>Go to <strong><a href="<?php echo esc_url(admin_url('admin.php?page=wpg-policies')); ?>">WP Governance → Policies</a></strong> to see all active policies. Each policy has a toggle switch — you can turn any policy on or off instantly.</p>
                    </div>
                </div>
                <div class="wpg-docs-step">
                    <div class="wpg-step-number">3</div>
                    <div>
                        <h4>Run a Test Simulation</h4>
                        <p>Go to the <strong><a href="<?php echo esc_url(admin_url('admin.php?page=wpg-dashboard')); ?>">Dashboard</a></strong> and click the <strong>🧪 Run Test Scenarios</strong> button. This fires 8 realistic AI scenarios through the full governance pipeline and logs the results so you can see enforcement in action.</p>
                    </div>
                </div>
            </div>

            <h3>Key Concepts</h3>
            <table class="wpg-docs-table">
                <tr>
                    <td><strong>Policy</strong></td>
                    <td>A rule that defines what happens when an AI agent tries to perform a specific action. Policies either <span class="wpg-badge wpg-badge-deny">DENY</span> or <span class="wpg-badge wpg-badge-allow">ALLOW</span> the action.</td>
                </tr>
                <tr>
                    <td><strong>Ability</strong></td>
                    <td>A WordPress action that an AI agent wants to perform, like <code>core/delete-posts</code> or <code>core/install-plugin</code>. Policies are matched against abilities.</td>
                </tr>
                <tr>
                    <td><strong>Condition</strong></td>
                    <td>An optional filter that makes a policy more specific. For example: "deny deletion only when item count is greater than 10."</td>
                </tr>
                <tr>
                    <td><strong>Audit Log</strong></td>
                    <td>An immutable record of every governance decision — every allow and every deny. Includes timestamps, agent names, policies matched, and tamper-evident checksums.</td>
                </tr>
                <tr>
                    <td><strong>Gate</strong></td>
                    <td>The enforcement point. Every AI ability request passes through the gate, which evaluates it against all active policies in priority order.</td>
                </tr>
                <tr>
                    <td><strong>Priority</strong></td>
                    <td>A number from 1 to 100 that determines the order policies are checked. <strong>Lower numbers = checked first.</strong> The first matching policy wins.</td>
                </tr>
            </table>
        </div>
    </div>

    <!-- ==================== HOW IT WORKS ==================== -->
    <div class="wpg-docs-panel" id="tab-how-it-works">
        <div class="wpg-card">
            <h2>⚙️ How the Governance Engine Works</h2>

            <h3>The Evaluation Pipeline</h3>
            <p>When an AI agent requests an action, the governance gate evaluates it in this order:</p>

            <div class="wpg-docs-flow">
                <div class="wpg-flow-step">
                    <div class="wpg-flow-icon">📨</div>
                    <h4>1. Request Received</h4>
                    <p>The AI agent sends an MCP request with an <strong>ability name</strong> (e.g., <code>core/delete-posts</code>) and <strong>arguments</strong> (e.g., <code>{ "count": 50 }</code>).</p>
                </div>
                <div class="wpg-flow-arrow">→</div>
                <div class="wpg-flow-step">
                    <div class="wpg-flow-icon">🔍</div>
                    <h4>2. Policy Matching</h4>
                    <p>The engine loops through all <strong>active policies</strong> in priority order (lowest number first). For each policy, it checks:<br>
                    ① Does the ability match?<br>
                    ② Does the agent match (if specified)?<br>
                    ③ Do all conditions pass?</p>
                </div>
                <div class="wpg-flow-arrow">→</div>
                <div class="wpg-flow-step">
                    <div class="wpg-flow-icon">⚖️</div>
                    <h4>3. Decision</h4>
                    <p>The <strong>first matching policy wins</strong>. If its decision is <strong>deny</strong>, the action is blocked. If <strong>allow</strong>, it proceeds. If no policy matches, the default is <strong>allow</strong>.</p>
                </div>
                <div class="wpg-flow-arrow">→</div>
                <div class="wpg-flow-step">
                    <div class="wpg-flow-icon">📝</div>
                    <h4>4. Audit Logged</h4>
                    <p>Every decision is recorded in the audit log with a tamper-evident checksum. This creates an immutable trail of AI governance on your site.</p>
                </div>
            </div>

            <h3>Design Principles</h3>
            <div class="wpg-docs-principles">
                <div class="wpg-principle">
                    <h4>🚫 Zero LLM in the Loop</h4>
                    <p>The governance engine is 100% deterministic. It uses pattern matching, not AI, to evaluate policies. This means decisions are instant, predictable, and impossible to prompt-inject.</p>
                </div>
                <div class="wpg-principle">
                    <h4>🌐 Zero External API Calls</h4>
                    <p>All evaluation happens locally on your server. No data leaves your site. No third-party dependencies. No internet connection required for governance.</p>
                </div>
                <div class="wpg-principle">
                    <h4>⚡ Sub-Millisecond Speed</h4>
                    <p>Policy evaluation typically completes in under 1 millisecond. AI agents won't experience any noticeable delay from governance checks.</p>
                </div>
                <div class="wpg-principle">
                    <h4>🔒 Append-Only Audit Log</h4>
                    <p>Once written, audit entries cannot be modified or deleted through the plugin. Each entry includes a cryptographic checksum for tamper detection.</p>
                </div>
            </div>
        </div>
    </div>

    <!-- ==================== WRITING POLICIES ==================== -->
    <div class="wpg-docs-panel" id="tab-writing-policies">
        <div class="wpg-card">
            <h2>📝 Writing Policies — Step by Step</h2>

            <p>You can create custom policies directly from the <strong><a href="<?php echo esc_url(admin_url('admin.php?page=wpg-policies')); ?>">Policies page</a></strong> using the visual editor. Here's what each field means:</p>

            <h3>Policy Fields Explained</h3>

            <div class="wpg-docs-field">
                <h4>📛 Policy Name <span class="required">*required</span></h4>
                <p>A short, descriptive name for your policy. Use clear action-oriented names like:</p>
                <ul>
                    <li>✅ <code>No Bulk Content Deletion</code></li>
                    <li>✅ <code>Block Theme Changes</code></li>
                    <li>❌ <code>Policy 1</code> (too vague)</li>
                </ul>
            </div>

            <div class="wpg-docs-field">
                <h4>📄 Description</h4>
                <p>A longer explanation of what this policy does and why. This is for your reference and appears in the policy list.</p>
            </div>

            <div class="wpg-docs-field">
                <h4>⚖️ Decision</h4>
                <p>What happens when this policy matches:</p>
                <ul>
                    <li><span class="wpg-badge wpg-badge-deny">DENY</span> — Block the action and return an error to the AI agent</li>
                    <li><span class="wpg-badge wpg-badge-allow">ALLOW</span> — Explicitly permit the action (useful for creating exceptions to broader deny rules)</li>
                </ul>
            </div>

            <div class="wpg-docs-field">
                <h4>🔢 Priority (1–100)</h4>
                <p><strong>Lower numbers are checked first.</strong> When multiple policies could match the same ability, the one with the lowest priority number wins.</p>
                <table class="wpg-docs-table">
                    <tr><td><strong>1–10</strong></td><td>Critical security policies (admin creation, file editing)</td></tr>
                    <tr><td><strong>11–30</strong></td><td>Important operational policies (plugin management, settings)</td></tr>
                    <tr><td><strong>31–60</strong></td><td>Content governance policies (post creation, moderation)</td></tr>
                    <tr><td><strong>61–100</strong></td><td>Low-priority or catch-all policies</td></tr>
                </table>
            </div>

            <div class="wpg-docs-field">
                <h4>🎯 Abilities</h4>
                <p>Select which AI actions this policy applies to. You can check boxes for common abilities, or type custom ones. Patterns supported:</p>
                <table class="wpg-docs-table">
                    <tr><td><code>core/delete-posts</code></td><td>Matches this exact ability only</td></tr>
                    <tr><td><code>core/*</code></td><td>Matches ALL abilities in the <code>core</code> namespace</td></tr>
                    <tr><td><code>*</code></td><td>Matches ALL abilities (careful!)</td></tr>
                </table>
            </div>

            <div class="wpg-docs-field">
                <h4>💬 Message When Blocked</h4>
                <p>The message returned to the AI agent when this policy denies an action. You can use template variables:</p>
                <table class="wpg-docs-table">
                    <tr><td><code>{ability}</code></td><td>The ability name, e.g., <code>core/delete-posts</code></td></tr>
                    <tr><td><code>{agent}</code></td><td>The AI agent name, e.g., <code>Claude Desktop</code></td></tr>
                    <tr><td><code>{args.field}</code></td><td>Any field from the arguments, e.g., <code>{args.role}</code></td></tr>
                </table>
                <p><strong>Example:</strong> <em>"AI agent {agent} is not allowed to perform {ability}. Contact the site administrator."</em></p>
            </div>

            <h3>Example: Block AI from Publishing Posts</h3>
            <div class="wpg-docs-example">
                <table class="wpg-docs-table">
                    <tr><td><strong>Name</strong></td><td>No AI Publishing</td></tr>
                    <tr><td><strong>Decision</strong></td><td>🛑 Deny</td></tr>
                    <tr><td><strong>Priority</strong></td><td>30</td></tr>
                    <tr><td><strong>Abilities</strong></td><td>☑ Publish Post</td></tr>
                    <tr><td><strong>Message</strong></td><td>AI agents cannot publish posts directly. Please save as draft and a human editor will review.</td></tr>
                    <tr><td><strong>Conditions</strong></td><td><em>None — applies every time</em></td></tr>
                </table>
            </div>

            <h3>Example: Allow Drafts but Block Publishing</h3>
            <div class="wpg-docs-example">
                <p>This requires two policies working together:</p>
                <p><strong>Policy 1 (Priority 25):</strong> Allow <code>core/create-post</code> when <code>args.post_status</code> equals <code>draft</code></p>
                <p><strong>Policy 2 (Priority 30):</strong> Deny <code>core/create-post</code> when <code>args.post_status</code> equals <code>publish</code></p>
                <p>Because Policy 1 has a lower priority number, it's checked first — drafts are allowed before the deny rule is reached.</p>
            </div>
        </div>
    </div>

    <!-- ==================== CONDITIONS REFERENCE ==================== -->
    <div class="wpg-docs-panel" id="tab-conditions">
        <div class="wpg-card">
            <h2>🔧 Conditions Reference</h2>

            <p>Conditions make policies more specific. Without conditions, a policy applies every time its abilities are matched. With conditions, it only triggers when <strong>all</strong> conditions are met.</p>

            <div class="wpg-docs-callout wpg-docs-callout-info">
                <strong>💡 Tip:</strong> Use the visual condition builder on the Policies page — no JSON required! Select a field, choose an operator, and enter a value.
            </div>

            <h3>Available Operators</h3>
            <table class="wpg-docs-table wpg-docs-table-full">
                <thead>
                    <tr><th>Operator</th><th>Description</th><th>Example Use Case</th></tr>
                </thead>
                <tbody>
                    <tr><td><code>equals</code></td><td>Exact match</td><td>Block when role equals "administrator"</td></tr>
                    <tr><td><code>not_equals</code></td><td>Anything except this value</td><td>Allow everything except "administrator" role</td></tr>
                    <tr><td><code>greater_than</code></td><td>Numeric: value is more than</td><td>Block when deleting more than 10 items</td></tr>
                    <tr><td><code>less_than</code></td><td>Numeric: value is less than</td><td>Allow when count is under 5</td></tr>
                    <tr><td><code>greater_than_or_equal</code></td><td>Numeric: at least this value</td><td>Block when editing 20+ pages at once</td></tr>
                    <tr><td><code>less_than_or_equal</code></td><td>Numeric: at most this value</td><td>Allow up to 3 post creations</td></tr>
                    <tr><td><code>contains</code></td><td>Text includes substring</td><td>Block files containing "wp-config"</td></tr>
                    <tr><td><code>not_contains</code></td><td>Text does not include</td><td>Allow file edits not containing ".php"</td></tr>
                    <tr><td><code>starts_with</code></td><td>Text begins with</td><td>Block options starting with "siteurl"</td></tr>
                    <tr><td><code>ends_with</code></td><td>Text ends with</td><td>Block files ending with ".php"</td></tr>
                    <tr><td><code>in</code></td><td>Value is one of a list</td><td>Block when role is in [administrator, editor]</td></tr>
                    <tr><td><code>exists</code></td><td>Field is present (any value)</td><td>Block when "force" flag is present</td></tr>
                    <tr><td><code>not_exists</code></td><td>Field is missing</td><td>Block when "confirmation" field is missing</td></tr>
                </tbody>
            </table>

            <h3>Common Field Paths</h3>
            <table class="wpg-docs-table wpg-docs-table-full">
                <thead>
                    <tr><th>Field</th><th>What It Contains</th><th>Example Values</th></tr>
                </thead>
                <tbody>
                    <tr><td><code>args.role</code></td><td>User role being assigned</td><td>administrator, editor, author, subscriber</td></tr>
                    <tr><td><code>args.count</code></td><td>Number of items affected</td><td>1, 10, 50</td></tr>
                    <tr><td><code>args.post_type</code></td><td>Content type being created</td><td>post, page, attachment</td></tr>
                    <tr><td><code>args.post_status</code></td><td>Post publication status</td><td>draft, publish, pending, private</td></tr>
                    <tr><td><code>args.option_name</code></td><td>WordPress option being changed</td><td>siteurl, blogname, users_can_register</td></tr>
                    <tr><td><code>args.file</code></td><td>File path being accessed</td><td>wp-config.php, .htaccess</td></tr>
                    <tr><td><code>args.slug</code></td><td>Plugin or theme identifier</td><td>akismet, twentytwentyfive</td></tr>
                    <tr><td><code>agent</code></td><td>Name of the AI agent</td><td>Claude Desktop, Cursor AI, ChatGPT</td></tr>
                    <tr><td><code>agent_type</code></td><td>Type of agent connection</td><td>mcp, simulation, manual</td></tr>
                </tbody>
            </table>
        </div>
    </div>

    <!-- ==================== BUILT-IN POLICIES ==================== -->
    <div class="wpg-docs-panel" id="tab-builtin-policies">
        <div class="wpg-card">
            <h2>🛡️ Built-in Policies</h2>

            <p>These 6 policies are installed automatically when the plugin is activated. They represent best-practice security defaults. You can edit or disable them from the <a href="<?php echo esc_url(admin_url('admin.php?page=wpg-policies')); ?>">Policies page</a>.</p>

            <div class="wpg-docs-policy-card">
                <h4>🔴 No Admin User Creation <span class="wpg-priority">P5</span></h4>
                <p>Prevents AI agents from creating new administrator accounts or promoting existing users to the administrator role. Admin creation must always involve human authorization.</p>
                <p><strong>Abilities:</strong> <code>core/create-user</code>, <code>core/update-user-role</code></p>
                <p><strong>Condition:</strong> Only when role is "administrator" or "admin"</p>
            </div>

            <div class="wpg-docs-policy-card">
                <h4>🔴 No File Editing <span class="wpg-priority">P5</span></h4>
                <p>Blocks AI agents from editing PHP files, theme files, or plugin code. Direct file modification is one of the highest-risk AI actions.</p>
                <p><strong>Abilities:</strong> <code>core/edit-file</code>, <code>core/write-file</code>, <code>core/theme-editor</code>, <code>core/plugin-editor</code></p>
            </div>

            <div class="wpg-docs-policy-card">
                <h4>🔴 No Plugin Installation <span class="wpg-priority">P5</span></h4>
                <p>Prevents AI agents from installing, activating, deactivating, or deleting plugins. This protects the site's attack surface from unauthorized code execution.</p>
                <p><strong>Abilities:</strong> <code>core/install-plugin</code>, <code>core/activate-plugin</code>, <code>core/deactivate-plugin</code>, <code>core/delete-plugin</code></p>
            </div>

            <div class="wpg-docs-policy-card">
                <h4>🔴 No Critical Settings Changes <span class="wpg-priority">P5</span></h4>
                <p>Blocks AI agents from modifying critical WordPress options like the site URL, registration settings, and other core configuration values.</p>
                <p><strong>Abilities:</strong> <code>core/update-option</code>, <code>core/update-settings</code></p>
                <p><strong>Condition:</strong> Only for critical options (siteurl, home, blogname, admin_email, users_can_register, default_role)</p>
            </div>

            <div class="wpg-docs-policy-card">
                <h4>🔴 No Bulk Deletion <span class="wpg-priority">P10</span></h4>
                <p>Prevents mass content deletion by AI agents. Deleting more than 10 items at once requires human approval.</p>
                <p><strong>Abilities:</strong> <code>core/delete-posts</code>, <code>core/delete-pages</code>, <code>core/delete-comments</code></p>
                <p><strong>Condition:</strong> Only when <code>args.count</code> is greater than 10</p>
            </div>

            <div class="wpg-docs-policy-card wpg-docs-policy-card-allow">
                <h4>🟢 Content Moderation <span class="wpg-priority">P20</span></h4>
                <p>Allows AI agents to create draft posts, pages, and manage content at a reasonable scale. This is the permission boundary for normal content operations.</p>
                <p><strong>Abilities:</strong> <code>core/create-post</code>, <code>core/create-page</code>, <code>core/update-post</code></p>
            </div>
        </div>
    </div>

    <!-- ==================== ABILITIES REFERENCE ==================== -->
    <div class="wpg-docs-panel" id="tab-abilities">
        <div class="wpg-card">
            <h2>📋 Abilities Reference</h2>
            <p>These are the standard WordPress MCP abilities that AI agents can request. Use these names when creating policies.</p>

            <h3>Content Abilities</h3>
            <table class="wpg-docs-table wpg-docs-table-full">
                <thead><tr><th>Ability</th><th>Description</th><th>Risk Level</th></tr></thead>
                <tbody>
                    <tr><td><code>core/create-post</code></td><td>Create a new blog post</td><td>🟡 Medium</td></tr>
                    <tr><td><code>core/update-post</code></td><td>Edit an existing post</td><td>🟡 Medium</td></tr>
                    <tr><td><code>core/delete-posts</code></td><td>Delete one or more posts</td><td>🔴 High</td></tr>
                    <tr><td><code>core/publish-post</code></td><td>Publish a draft post</td><td>🟡 Medium</td></tr>
                    <tr><td><code>core/create-page</code></td><td>Create a new page</td><td>🟡 Medium</td></tr>
                    <tr><td><code>core/delete-pages</code></td><td>Delete one or more pages</td><td>🔴 High</td></tr>
                    <tr><td><code>core/delete-comments</code></td><td>Delete comments</td><td>🟡 Medium</td></tr>
                    <tr><td><code>core/get-page</code></td><td>Read a page (non-destructive)</td><td>🟢 Low</td></tr>
                </tbody>
            </table>

            <h3>User & Security Abilities</h3>
            <table class="wpg-docs-table wpg-docs-table-full">
                <thead><tr><th>Ability</th><th>Description</th><th>Risk Level</th></tr></thead>
                <tbody>
                    <tr><td><code>core/create-user</code></td><td>Create a new user account</td><td>🔴 Critical</td></tr>
                    <tr><td><code>core/update-user-role</code></td><td>Change a user's role</td><td>🔴 Critical</td></tr>
                    <tr><td><code>core/delete-user</code></td><td>Delete a user account</td><td>🔴 High</td></tr>
                </tbody>
            </table>

            <h3>Infrastructure Abilities</h3>
            <table class="wpg-docs-table wpg-docs-table-full">
                <thead><tr><th>Ability</th><th>Description</th><th>Risk Level</th></tr></thead>
                <tbody>
                    <tr><td><code>core/install-plugin</code></td><td>Install a new plugin</td><td>🔴 Critical</td></tr>
                    <tr><td><code>core/activate-plugin</code></td><td>Activate a plugin</td><td>🔴 High</td></tr>
                    <tr><td><code>core/deactivate-plugin</code></td><td>Deactivate a plugin</td><td>🟡 Medium</td></tr>
                    <tr><td><code>core/delete-plugin</code></td><td>Delete a plugin</td><td>🔴 High</td></tr>
                    <tr><td><code>core/update-plugin</code></td><td>Update a plugin version</td><td>🔴 High</td></tr>
                    <tr><td><code>core/install-theme</code></td><td>Install a new theme</td><td>🔴 High</td></tr>
                    <tr><td><code>core/delete-theme</code></td><td>Delete a theme</td><td>🔴 High</td></tr>
                    <tr><td><code>core/update-option</code></td><td>Change a WordPress setting</td><td>🔴 Critical</td></tr>
                    <tr><td><code>core/update-settings</code></td><td>Modify site settings</td><td>🔴 Critical</td></tr>
                </tbody>
            </table>

            <h3>File Abilities</h3>
            <table class="wpg-docs-table wpg-docs-table-full">
                <thead><tr><th>Ability</th><th>Description</th><th>Risk Level</th></tr></thead>
                <tbody>
                    <tr><td><code>core/edit-file</code></td><td>Edit any file on the server</td><td>🔴 Critical</td></tr>
                    <tr><td><code>core/write-file</code></td><td>Write/create files</td><td>🔴 Critical</td></tr>
                    <tr><td><code>core/theme-editor</code></td><td>Use the theme file editor</td><td>🔴 Critical</td></tr>
                    <tr><td><code>core/plugin-editor</code></td><td>Use the plugin file editor</td><td>🔴 Critical</td></tr>
                </tbody>
            </table>
        </div>
    </div>

    <!-- ==================== REST API ==================== -->
    <div class="wpg-docs-panel" id="tab-rest-api">
        <div class="wpg-card">
            <h2>🔌 REST API Reference</h2>
            <p>All endpoints require the <code>manage_options</code> capability (administrator role) and a valid WP REST nonce.</p>

            <p><strong>Base URL:</strong> <code><?php echo esc_html(rest_url('wpg/v1/')); ?></code></p>

            <h3>Policy Endpoints</h3>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-allow">GET</span> <code>/wpg/v1/policies</code></h4>
                <p>List all policies.</p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-allow">GET</span> <code>/wpg/v1/policies/{id}</code></h4>
                <p>Get a single policy by ID.</p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-custom">POST</span> <code>/wpg/v1/policies</code></h4>
                <p>Create a new policy. Send the full policy object in the JSON body.</p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-warning">PUT</span> <code>/wpg/v1/policies/{id}</code></h4>
                <p>Update an existing policy.</p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-deny">DELETE</span> <code>/wpg/v1/policies/{id}</code></h4>
                <p>Delete a custom policy (built-in policies cannot be deleted).</p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-custom">POST</span> <code>/wpg/v1/policies/{id}/toggle</code></h4>
                <p>Toggle a policy on or off. Body: <code>{ "active": true|false }</code></p>
            </div>

            <h3>Governance Endpoints</h3>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-custom">POST</span> <code>/wpg/v1/check</code></h4>
                <p><strong>Dry run</strong> — evaluates an action against policies without logging. Body: <code>{ "ability": "core/delete-posts", "agent": "test" }</code></p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-custom">POST</span> <code>/wpg/v1/simulate</code></h4>
                <p><strong>Full pipeline</strong> — evaluates and writes to the audit log. Same body as <code>/check</code>.</p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-custom">POST</span> <code>/wpg/v1/simulate/batch</code></h4>
                <p>Runs 8 pre-configured test scenarios through the full pipeline. No body required.</p>
            </div>

            <h3>Audit Log Endpoints</h3>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-allow">GET</span> <code>/wpg/v1/audit-log</code></h4>
                <p>Query audit entries. Supports filters: <code>decision</code>, <code>ability</code>, <code>agent</code>, <code>per_page</code>, <code>page</code>, <code>order</code>.</p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-allow">GET</span> <code>/wpg/v1/audit-log/stats</code></h4>
                <p>Get summary statistics. Query param: <code>period=day|week|month|all</code>.</p>
            </div>

            <h3>Settings Endpoints</h3>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-allow">GET</span> <code>/wpg/v1/settings</code></h4>
                <p>Get current plugin settings.</p>
            </div>
            <div class="wpg-docs-endpoint">
                <h4><span class="wpg-badge wpg-badge-warning">PUT</span> <code>/wpg/v1/settings</code></h4>
                <p>Update plugin settings.</p>
            </div>
        </div>
    </div>

    <!-- ==================== TROUBLESHOOTING ==================== -->
    <div class="wpg-docs-panel" id="tab-troubleshooting">
        <div class="wpg-card">
            <h2>🔍 Troubleshooting</h2>

            <div class="wpg-docs-troubleshoot">
                <h4>❓ The Dashboard shows zero actions, but I've activated the plugin</h4>
                <p>The audit log only records events when AI agents actually interact with your site through MCP. To populate the dashboard with test data, click the <strong>🧪 Run Test Scenarios</strong> button which simulates 8 realistic scenarios.</p>
            </div>

            <div class="wpg-docs-troubleshoot">
                <h4>❓ A policy isn't blocking the action I expected</h4>
                <ol>
                    <li>Check the policy is <strong>toggled ON</strong> (green switch) on the Policies page</li>
                    <li>Verify the <strong>ability name</strong> matches exactly. Go to the Audit Log to see what ability name the agent is actually sending</li>
                    <li>Check the <strong>priority</strong>. If a higher-priority (lower number) ALLOW policy matches first, it will permit the action before your deny rule is reached</li>
                    <li>Check your <strong>conditions</strong>. If conditions are set but the incoming request doesn't have the matching arguments, the policy won't trigger</li>
                    <li>Use the <strong>Test Policy (Dry Run)</strong> button on the Policies page to quickly test an ability</li>
                </ol>
            </div>

            <div class="wpg-docs-troubleshoot">
                <h4>❓ I accidentally disabled a built-in policy. How do I restore it?</h4>
                <p>Built-in policies can be toggled back on from the Policies page. They cannot be deleted — only custom policies can be removed. If a built-in policy was edited unfavorably, deactivate and reactivate the plugin to restore defaults (note: this only works if no policies exist in the database).</p>
            </div>

            <div class="wpg-docs-troubleshoot">
                <h4>❓ The plugin shows a PHP error on activation</h4>
                <p>Ensure your server meets the requirements:</p>
                <ul>
                    <li>PHP 8.1 or higher</li>
                    <li>WordPress 6.0 or higher</li>
                    <li>MySQL 5.7+ or MariaDB 10.3+</li>
                </ul>
                <p>If you see a "class not found" error, ensure all plugin files were uploaded completely, including the <code>includes/</code> directory.</p>
            </div>

            <div class="wpg-docs-troubleshoot">
                <h4>❓ Audit log entries have mismatched checksums</h4>
                <p>This indicates the audit log entry was modified directly in the database outside of the plugin. This is a security feature — the checksum proves whether entries are authentic. Investigate any unauthorized database modifications.</p>
            </div>
        </div>
    </div>

    <!-- ==================== FAQ ==================== -->
    <div class="wpg-docs-panel" id="tab-faq">
        <div class="wpg-card">
            <h2>❓ Frequently Asked Questions</h2>

            <div class="wpg-docs-faq">
                <h4>Does this plugin slow down my site?</h4>
                <p>No. The governance gate only activates when an AI agent sends an MCP request. Normal website visitors, search engines, and manual admin actions are completely unaffected. When the gate does activate, evaluation takes less than 1 millisecond.</p>
            </div>

            <div class="wpg-docs-faq">
                <h4>Does this work with Claude, ChatGPT, Copilot, Cursor, etc.?</h4>
                <p>Yes. WP Governance works with any AI agent that connects to WordPress via the Model Context Protocol (MCP). This includes Claude Desktop, ChatGPT with plugins, VS Code Copilot, Cursor, Windsurf, Devin, and any future MCP-compatible agents.</p>
            </div>

            <div class="wpg-docs-faq">
                <h4>Can an AI agent bypass the governance gate?</h4>
                <p>No. The gate intercepts ALL MCP ability requests at the WordPress hook level before the action is executed. An AI agent cannot bypass the gate unless it has direct database access outside of WordPress — which is a server security issue, not a plugin issue.</p>
            </div>

            <div class="wpg-docs-faq">
                <h4>Does the plugin send data to external servers?</h4>
                <p><strong>No.</strong> All policy evaluation happens entirely on your server. Zero external API calls. Zero telemetry. Zero data collection. Your governance policies and audit logs stay on your database.</p>
            </div>

            <div class="wpg-docs-faq">
                <h4>What's the difference between "Check" (dry run) and "Simulate"?</h4>
                <p><strong>Check</strong> evaluates a policy and returns the result, but does NOT write to the audit log. <strong>Simulate</strong> evaluates AND writes to the audit log. Use Check for quick testing; use Simulate for full pipeline verification.</p>
            </div>

            <div class="wpg-docs-faq">
                <h4>Can I create policies for custom MCP tools?</h4>
                <p>Yes. Abilities can be any string, not just the built-in <code>core/*</code> ones. If your MCP server exposes custom tools like <code>myapp/send-email</code> or <code>analytics/export-data</code>, you can create policies that match those ability names.</p>
            </div>

            <div class="wpg-docs-faq">
                <h4>What happens if I have conflicting policies?</h4>
                <p>The policy with the <strong>lowest priority number</strong> wins. If two policies both match the same ability, the one checked first (lower priority number) makes the decision. The second policy is never reached. Design your priority numbers carefully to avoid conflicts.</p>
            </div>

            <div class="wpg-docs-faq">
                <h4>Is the audit log GDPR compliant?</h4>
                <p>The audit log records AI agent identifiers and IP addresses for security purposes. Since AI agents are not natural persons, GDPR typically does not apply to agent activity logs. However, user IDs of the WordPress account that authenticated the agent session may be considered personal data. Consult your data protection officer for your specific jurisdiction.</p>
            </div>
        </div>
    </div>
</div>

<!-- Documentation Tab Switcher -->
<script>
(function() {
    const tabs = document.querySelectorAll('.wpg-docs-tab');
    const panels = document.querySelectorAll('.wpg-docs-panel');

    tabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const target = this.dataset.tab;

            tabs.forEach(t => t.classList.remove('active'));
            panels.forEach(p => p.classList.remove('active'));

            this.classList.add('active');
            document.getElementById('tab-' + target).classList.add('active');

            // Update URL hash for direct linking
            history.replaceState(null, '', '#' + target);
        });
    });

    // Handle direct linking via hash
    const hash = window.location.hash.replace('#', '');
    if (hash) {
        const tab = document.querySelector(`.wpg-docs-tab[data-tab="${hash}"]`);
        if (tab) tab.click();
    }
})();
</script>
