<?php
/**
 * WPG Admin Policies View
 *
 * @package aos-wp-governance
 * @since   1.0.0
 */

defined('ABSPATH') || exit;

$all_policies = get_option('wpg_policies', []);
?>

<div class="wrap wpg-wrap">
    <h1 class="wpg-title">
        <span class="dashicons dashicons-shield"></span>
        <?php esc_html_e('Governance Policies', 'aos-wp-governance'); ?>
    </h1>

    <p class="wpg-subtitle">
        <?php esc_html_e('Manage the constitutional rules that govern AI agent behavior on this site.', 'aos-wp-governance'); ?>
    </p>

    <div class="wpg-toolbar">
        <button type="button" class="button button-primary" id="wpg-add-policy">
            <span class="dashicons dashicons-plus-alt2"></span>
            <?php esc_html_e('Add Custom Policy', 'aos-wp-governance'); ?>
        </button>
        <button type="button" class="button" id="wpg-test-policy">
            <span class="dashicons dashicons-superhero-alt"></span>
            <?php esc_html_e('Test Policy (Dry Run)', 'aos-wp-governance'); ?>
        </button>
    </div>

    <!-- Policies Table -->
    <div class="wpg-card">
        <?php if (empty($all_policies)): ?>
            <div class="wpg-empty">
                <h3><?php esc_html_e('No policies configured', 'aos-wp-governance'); ?></h3>
                <p><?php esc_html_e('Default policies will be installed automatically. You can also create custom policies.', 'aos-wp-governance'); ?></p>
            </div>
        <?php
else: ?>
            <table class="wpg-table wpg-table-policies">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Status', 'aos-wp-governance'); ?></th>
                        <th><?php esc_html_e('Priority', 'aos-wp-governance'); ?></th>
                        <th><?php esc_html_e('Name', 'aos-wp-governance'); ?></th>
                        <th><?php esc_html_e('Decision', 'aos-wp-governance'); ?></th>
                        <th><?php esc_html_e('Abilities', 'aos-wp-governance'); ?></th>
                        <th><?php esc_html_e('Source', 'aos-wp-governance'); ?></th>
                        <th><?php esc_html_e('Actions', 'aos-wp-governance'); ?></th>
                    </tr>
                </thead>
                <tbody id="wpg-policies-list">
                    <?php foreach ($all_policies as $policy): ?>
                        <tr data-policy-id="<?php echo esc_attr($policy['id']); ?>"
                            class="<?php echo($policy['active'] ?? true) ? '' : 'wpg-row-inactive'; ?>">
                            <td>
                                <label class="wpg-toggle">
                                    <input type="checkbox"
                                        class="wpg-policy-toggle"
                                        data-id="<?php echo esc_attr($policy['id']); ?>"
                                        <?php checked($policy['active'] ?? true); ?>>
                                    <span class="wpg-toggle-slider"></span>
                                </label>
                            </td>
                            <td>
                                <span class="wpg-priority">P<?php echo esc_html($policy['priority'] ?? 100); ?></span>
                            </td>
                            <td>
                                <strong><?php echo esc_html($policy['name']); ?></strong>
                                <br>
                                <small class="wpg-text-muted"><?php echo esc_html($policy['description'] ?? ''); ?></small>
                            </td>
                            <td>
                                <span class="wpg-badge wpg-badge-<?php echo esc_attr($policy['decision'] ?? 'deny'); ?>">
                                    <?php echo esc_html(strtoupper($policy['decision'] ?? 'DENY')); ?>
                                </span>
                            </td>
                            <td>
                                <?php
        $abilities = $policy['abilities'] ?? ['*'];
        foreach (array_slice($abilities, 0, 3) as $ability) {
            printf('<code class="wpg-ability">%s</code> ', esc_html($ability));
        }
        if (count($abilities) > 3) {
            printf(
                '<span class="wpg-text-muted">+%d more</span>',
                count($abilities) - 3
            );
        }
?>
                            </td>
                            <td>
                                <?php if (str_starts_with($policy['id'], 'wpg-builtin-')): ?>
                                    <span class="wpg-badge wpg-badge-builtin"><?php esc_html_e('Built-in', 'aos-wp-governance'); ?></span>
                                <?php
        else: ?>
                                    <span class="wpg-badge wpg-badge-custom"><?php esc_html_e('Custom', 'aos-wp-governance'); ?></span>
                                <?php
        endif; ?>
                            </td>
                            <td>
                                <button type="button" class="button button-small wpg-edit-policy"
                                    data-id="<?php echo esc_attr($policy['id']); ?>">
                                    <?php esc_html_e('Edit', 'aos-wp-governance'); ?>
                                </button>
                                <?php if (!str_starts_with($policy['id'], 'wpg-builtin-')): ?>
                                    <button type="button" class="button button-small button-link-delete wpg-delete-policy"
                                        data-id="<?php echo esc_attr($policy['id']); ?>">
                                        <?php esc_html_e('Delete', 'aos-wp-governance'); ?>
                                    </button>
                                <?php
        endif; ?>
                            </td>
                        </tr>
                    <?php
    endforeach; ?>
                </tbody>
            </table>
        <?php
endif; ?>
    </div>

    <!-- Policy Editor Modal (hidden by default, toggled by JS) -->
    <div id="wpg-policy-modal" class="wpg-modal" style="display:none;">
        <div class="wpg-modal-content" style="max-width: 680px;">
            <div class="wpg-modal-header">
                <h2 id="wpg-modal-title"><?php esc_html_e('Add Policy', 'aos-wp-governance'); ?></h2>
                <button type="button" class="wpg-modal-close">&times;</button>
            </div>
            <form id="wpg-policy-form">

                <!-- Step 1: Basics -->
                <div class="wpg-form-section">
                    <h3 class="wpg-form-section-title">
                        <span class="wpg-step-number">1</span>
                        <?php esc_html_e('Policy Basics', 'aos-wp-governance'); ?>
                        <span class="wpg-tooltip" title="<?php esc_attr_e('Give your policy a clear name and description so you can identify it later.', 'aos-wp-governance'); ?>">ⓘ</span>
                    </h3>
                    <div class="wpg-form-group">
                        <label for="wpg-policy-name"><?php esc_html_e('Policy Name', 'aos-wp-governance'); ?> <span class="required">*</span></label>
                        <input type="text" id="wpg-policy-name" name="name" required placeholder="e.g., Prevent Mass Content Deletion">
                    </div>
                    <div class="wpg-form-group">
                        <label for="wpg-policy-desc"><?php esc_html_e('Description', 'aos-wp-governance'); ?></label>
                        <textarea id="wpg-policy-desc" name="description" rows="2" placeholder="Describe what this policy protects against..."></textarea>
                    </div>
                </div>

                <!-- Step 2: Decision & Priority -->
                <div class="wpg-form-section">
                    <h3 class="wpg-form-section-title">
                        <span class="wpg-step-number">2</span>
                        <?php esc_html_e('Decision & Priority', 'aos-wp-governance'); ?>
                        <span class="wpg-tooltip" title="<?php esc_attr_e('Choose whether this policy blocks or allows actions. Lower priority numbers are evaluated first.', 'aos-wp-governance'); ?>">ⓘ</span>
                    </h3>
                    <div class="wpg-form-row">
                        <div class="wpg-form-group">
                            <label for="wpg-policy-decision"><?php esc_html_e('When matched, this policy should:', 'aos-wp-governance'); ?></label>
                            <select id="wpg-policy-decision" name="decision">
                                <option value="deny">🛑 Block the action (Deny)</option>
                                <option value="allow">✅ Allow the action (Allow)</option>
                            </select>
                        </div>
                        <div class="wpg-form-group" style="max-width: 140px;">
                            <label for="wpg-policy-priority"><?php esc_html_e('Priority', 'aos-wp-governance'); ?>
                                <span class="wpg-tooltip" title="<?php esc_attr_e('1 = highest priority (checked first), 100 = lowest. Policies are evaluated in order from lowest to highest number.', 'aos-wp-governance'); ?>">ⓘ</span>
                            </label>
                            <input type="number" id="wpg-policy-priority" name="priority" value="50" min="1" max="100">
                        </div>
                    </div>
                    <div class="wpg-form-group">
                        <label for="wpg-policy-reason"><?php esc_html_e('Message when action is blocked', 'aos-wp-governance'); ?>
                            <span class="wpg-tooltip" title="<?php esc_attr_e('This message is shown to the AI agent when its action is denied. You can use {ability} for the action name and {agent} for the AI agent name.', 'aos-wp-governance'); ?>">ⓘ</span>
                        </label>
                        <textarea id="wpg-policy-reason" name="reason" rows="2" placeholder="e.g., This action is not allowed. Please contact the site administrator."></textarea>
                    </div>
                </div>

                <!-- Step 3: Abilities -->
                <div class="wpg-form-section">
                    <h3 class="wpg-form-section-title">
                        <span class="wpg-step-number">3</span>
                        <?php esc_html_e('Which AI actions does this apply to?', 'aos-wp-governance'); ?>
                        <span class="wpg-tooltip" title="<?php esc_attr_e('Select which WordPress abilities (actions) this policy governs. You can choose from common ones below or add custom ability names.', 'aos-wp-governance'); ?>">ⓘ</span>
                    </h3>
                    <div class="wpg-ability-picker">
                        <div class="wpg-ability-group">
                            <strong><?php esc_html_e('Content', 'aos-wp-governance'); ?></strong>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/create-post"> Create Post</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/delete-posts"> Delete Posts</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/create-page"> Create Page</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/delete-pages"> Delete Pages</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/publish-post"> Publish Post</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/delete-comments"> Delete Comments</label>
                        </div>
                        <div class="wpg-ability-group">
                            <strong><?php esc_html_e('Users & Security', 'aos-wp-governance'); ?></strong>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/create-user"> Create User</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/update-user-role"> Change User Role</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/delete-user"> Delete User</label>
                        </div>
                        <div class="wpg-ability-group">
                            <strong><?php esc_html_e('Site Infrastructure', 'aos-wp-governance'); ?></strong>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/install-plugin"> Install Plugin</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/delete-plugin"> Delete Plugin</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/update-plugin"> Update Plugin</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/install-theme"> Install Theme</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/delete-theme"> Delete Theme</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/update-option"> Change Settings</label>
                        </div>
                        <div class="wpg-ability-group">
                            <strong><?php esc_html_e('Files', 'aos-wp-governance'); ?></strong>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/edit-file"> Edit File</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/write-file"> Write File</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/theme-editor"> Theme Editor</label>
                            <label><input type="checkbox" class="wpg-ability-check" value="core/plugin-editor"> Plugin Editor</label>
                        </div>
                    </div>
                    <div class="wpg-form-group" style="margin-top: 10px;">
                        <label for="wpg-policy-abilities-custom"><?php esc_html_e('Additional abilities (optional)', 'aos-wp-governance'); ?></label>
                        <input type="text" id="wpg-policy-abilities-custom" placeholder="e.g., custom/my-action, mcp/tool-name">
                        <small class="wpg-help"><?php esc_html_e('Add custom ability names separated by commas. Use * for ALL abilities or namespace/* for all actions in a namespace.', 'aos-wp-governance'); ?></small>
                    </div>
                    <!-- Hidden field populated by JS from checkboxes + custom -->
                    <input type="hidden" id="wpg-policy-abilities" name="abilities">
                </div>

                <!-- Step 4: Conditions (Visual Builder) -->
                <div class="wpg-form-section">
                    <h3 class="wpg-form-section-title">
                        <span class="wpg-step-number">4</span>
                        <?php esc_html_e('Conditions (optional)', 'aos-wp-governance'); ?>
                        <span class="wpg-tooltip" title="<?php esc_attr_e('Add conditions to make this policy more specific. Without conditions, the policy will apply to ALL matching abilities. With conditions, it only applies when ALL conditions are met.', 'aos-wp-governance'); ?>">ⓘ</span>
                    </h3>
                    <p class="wpg-help-block"><?php esc_html_e('Without conditions, this policy applies every time the selected abilities are used. Add conditions to only trigger on specific circumstances.', 'aos-wp-governance'); ?></p>

                    <div id="wpg-conditions-list">
                        <!-- Condition rows will be inserted here by JS -->
                    </div>

                    <button type="button" id="wpg-add-condition" class="button" style="margin-top: 8px;">
                        <span class="dashicons dashicons-plus-alt2" style="vertical-align: middle;"></span>
                        <?php esc_html_e('Add Condition', 'aos-wp-governance'); ?>
                    </button>

                    <!-- Hidden field populated by JS from visual builder -->
                    <input type="hidden" id="wpg-policy-conditions" name="conditions">
                </div>

                <input type="hidden" id="wpg-policy-id" name="id" value="">
                <div class="wpg-form-actions">
                    <button type="submit" class="button button-primary"><?php esc_html_e('Save Policy', 'aos-wp-governance'); ?></button>
                    <button type="button" class="button wpg-modal-close"><?php esc_html_e('Cancel', 'aos-wp-governance'); ?></button>
                </div>
            </form>
        </div>
    </div>
</div>
