<?php
/**
 * AOS WP Governance — Constitutional AI Governance for WordPress MCP
 *
 * @package     aos-wp-governance
 * @author      AOS Foundation
 * @copyright   2026 Salvatore Systems LLC
 * @license     GPL-2.0-or-later
 *
 * @wordpress-plugin
 * Plugin Name:       AOS WP Governance
 * Plugin URI:        https://wp-governance.com
 * Description:       Constitutional AI governance for WordPress MCP. Deterministic policy gates between AI agent intent and site execution. The AI doesn't decide — the gate enforces.
 * Requires at least: 6.0
 * Requires PHP:      8.1
 * Version:           1.0.0
 * Author:            AOS Foundation
 * Author URI:        https://aos-foundation.com
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * Text Domain:       aos-wp-governance
 * Domain Path:       /languages
 */

// Exit if accessed directly.
defined('ABSPATH') || exit;

/**
 * Plugin constants.
 */
define('WPG_VERSION', '1.0.0');
define('WPG_PLUGIN_FILE', __FILE__);
define('WPG_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WPG_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WPG_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('WPG_DB_VERSION', '1.0.0');
define('WPG_MIN_WP_VERSION', '6.0');
define('WPG_MIN_PHP_VERSION', '8.1');

/**
 * Check minimum requirements before loading the plugin.
 */
function wpg_check_requirements(): bool
{
    $errors = [];

    if (version_compare(PHP_VERSION, WPG_MIN_PHP_VERSION, '<')) {
        $errors[] = sprintf(
            /* translators: 1: Required PHP version, 2: Current PHP version */
            __('AOS WP Governance requires PHP %1$s or higher. You are running PHP %2$s.', 'aos-wp-governance'),
            WPG_MIN_PHP_VERSION,
            PHP_VERSION
        );
    }

    if (version_compare(get_bloginfo('version'), WPG_MIN_WP_VERSION, '<')) {
        $errors[] = sprintf(
            /* translators: 1: Required WP version, 2: Current WP version */
            __('AOS WP Governance requires WordPress %1$s or higher. You are running WordPress %2$s.', 'aos-wp-governance'),
            WPG_MIN_WP_VERSION,
            get_bloginfo('version')
        );
    }

    if (!empty($errors)) {
        add_action('admin_notices', function () use ($errors) {
            foreach ($errors as $error) {
                printf('<div class="notice notice-error"><p>%s</p></div>', esc_html($error));
            }
        });
        return false;
    }

    return true;
}

/**
 * Load plugin classes.
 */
function wpg_load_classes(): void
{
    // Load the governance engine first (WPG_Policy_Result + WPG_Policy_Engine).
    // This file is named differently from the convention because the standard
    // name 'class-wpg-policy-engine.php' triggered file system issues.
    require_once WPG_PLUGIN_DIR . 'includes/class-wpg-governance-engine.php';

    $classes = [
        'WPG_Audit_Log',
        'WPG_Agent_Detector',
        'WPG_Gate',
        'WPG_Abilities',
        'WPG_Admin',
        'WPG_Rest_API',
    ];

    foreach ($classes as $class) {
        $file = WPG_PLUGIN_DIR . 'includes/class-' . strtolower(str_replace('_', '-', $class)) . '.php';
        if (file_exists($file)) {
            require_once $file;
        }
    }
}

/**
 * Initialize the plugin.
 */
function wpg_init(): void
{
    if (!wpg_check_requirements()) {
        return;
    }

    wpg_load_classes();

    // Boot the core components.
    WPG_Audit_Log::instance();
    WPG_Policy_Engine::instance();
    WPG_Agent_Detector::instance();
    WPG_Gate::instance();
    WPG_Abilities::instance();
    WPG_Rest_API::instance();

    if (is_admin()) {
        WPG_Admin::instance();
    }

    /**
     * Fires after all AOS WP Governance components have been initialized.
     *
     * @since 1.0.0
     */
    do_action('wpg_loaded');
}

/**
 * Plugin activation callback.
 */
function wpg_activate(): void
{
    if (!wpg_check_requirements()) {
        return;
    }

    wpg_load_classes();

    // Create the audit log table.
    WPG_Audit_Log::instance()->create_table();

    // Install default policies.
    WPG_Policy_Engine::instance()->install_default_policies();

    // Set activation flag for welcome screen.
    set_transient('wpg_activated', true, 60);

    // Store the installed version.
    update_option('wpg_version', WPG_VERSION);
    update_option('wpg_db_version', WPG_DB_VERSION);

    // Flush rewrite rules for REST endpoints.
    flush_rewrite_rules();
}

/**
 * Plugin deactivation callback.
 */
function wpg_deactivate(): void
{
    // Clean up transients.
    delete_transient('wpg_activated');

    // Note: We intentionally do NOT remove the audit log table or policies.
    // That only happens on uninstall (uninstall.php).

    flush_rewrite_rules();
}

// Register activation/deactivation hooks.
register_activation_hook(WPG_PLUGIN_FILE, 'wpg_activate');
register_deactivation_hook(WPG_PLUGIN_FILE, 'wpg_deactivate');

// Boot the plugin after WordPress is fully loaded.
add_action('plugins_loaded', 'wpg_init');
