<?php
/**
 * WPG REST API — Plugin REST endpoints for the admin dashboard.
 *
 * @package aos-wp-governance
 * @since   1.0.0
 */

defined('ABSPATH') || exit;

class WPG_Rest_API
{

    /** @var WPG_Rest_API|null */
    private static ?WPG_Rest_API $instance = null;

    private const NAMESPACE = 'wpg/v1';

    public static function instance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    /**
     * Register REST API routes.
     */
    public function register_routes(): void
    {
        // Audit log endpoints.
        register_rest_route(self::NAMESPACE , '/audit-log', [
            'methods' => 'GET',
            'callback' => [$this, 'get_audit_log'],
            'permission_callback' => [$this, 'admin_permission_check'],
            'args' => [
                'decision' => ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field'],
                'ability' => ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field'],
                'agent' => ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field'],
                'from' => ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field'],
                'to' => ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field'],
                'per_page' => ['type' => 'integer', 'default' => 50],
                'page' => ['type' => 'integer', 'default' => 1],
            ],
        ]);

        register_rest_route(self::NAMESPACE , '/audit-log/stats', [
            'methods' => 'GET',
            'callback' => [$this, 'get_audit_stats'],
            'permission_callback' => [$this, 'admin_permission_check'],
            'args' => [
                'period' => ['type' => 'string', 'default' => 'day'],
            ],
        ]);

        // Policy endpoints.
        register_rest_route(self::NAMESPACE , '/policies', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_policies'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
            [
                'methods' => 'POST',
                'callback' => [$this, 'save_policy'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
        ]);

        register_rest_route(self::NAMESPACE , '/policies/(?P<id>[a-zA-Z0-9_-]+)', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_policy'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
            [
                'methods' => 'PUT',
                'callback' => [$this, 'update_policy'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
            [
                'methods' => 'DELETE',
                'callback' => [$this, 'delete_policy'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
        ]);

        register_rest_route(self::NAMESPACE , '/policies/(?P<id>[a-zA-Z0-9_-]+)/toggle', [
            'methods' => 'POST',
            'callback' => [$this, 'toggle_policy'],
            'permission_callback' => [$this, 'admin_permission_check'],
        ]);

        // Check policy (dry run).
        register_rest_route(self::NAMESPACE , '/check', [
            'methods' => 'POST',
            'callback' => [$this, 'check_policy'],
            'permission_callback' => [$this, 'admin_permission_check'],
        ]);

        // Simulate — full pipeline: evaluate + log.
        register_rest_route(self::NAMESPACE , '/simulate', [
            'methods' => 'POST',
            'callback' => [$this, 'simulate_action'],
            'permission_callback' => [$this, 'admin_permission_check'],
        ]);

        // Simulate batch — run preset test scenarios.
        register_rest_route(self::NAMESPACE , '/simulate/batch', [
            'methods' => 'POST',
            'callback' => [$this, 'simulate_batch'],
            'permission_callback' => [$this, 'admin_permission_check'],
        ]);

        // Settings.
        register_rest_route(self::NAMESPACE , '/settings', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_settings'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
            [
                'methods' => 'PUT',
                'callback' => [$this, 'update_settings'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
        ]);
    }

    // ===== Audit Log Endpoints =====

    public function get_audit_log(\WP_REST_Request $request): \WP_REST_Response
    {
        $result = WPG_Audit_Log::instance()->query($request->get_params());
        return new \WP_REST_Response($result, 200);
    }

    public function get_audit_stats(\WP_REST_Request $request): \WP_REST_Response
    {
        $period = $request->get_param('period') ?? 'day';
        $stats = WPG_Audit_Log::instance()->get_stats($period);
        return new \WP_REST_Response($stats, 200);
    }

    // ===== Policy Endpoints =====

    public function get_policies(\WP_REST_Request $request): \WP_REST_Response
    {
        $engine = WPG_Policy_Engine::instance();
        $policies = $engine->get_active_policies();

        // Include inactive policies too.
        $option = get_option('wpg_policies', []);

        return new \WP_REST_Response([
            'policies' => array_values($option),
            'total' => count($option),
        ], 200);
    }

    public function get_policy(\WP_REST_Request $request): \WP_REST_Response
    {
        $id = $request->get_param('id');
        $policy = WPG_Policy_Engine::instance()->get_policy($id);

        if (!$policy) {
            return new \WP_REST_Response(['error' => 'Policy not found.'], 404);
        }

        return new \WP_REST_Response($policy, 200);
    }

    public function save_policy(\WP_REST_Request $request): \WP_REST_Response
    {
        $body = $request->get_json_params();

        if (empty($body['name'])) {
            return new \WP_REST_Response(['error' => 'Policy name is required.'], 400);
        }

        $saved = WPG_Policy_Engine::instance()->save_policy($body);

        return new \WP_REST_Response([
            'success' => $saved,
            'policy' => $body,
        ], $saved ? 201 : 500);
    }

    public function update_policy(\WP_REST_Request $request): \WP_REST_Response
    {
        $id = $request->get_param('id');
        $body = $request->get_json_params();

        $body['id'] = $id;
        $saved = WPG_Policy_Engine::instance()->save_policy($body);

        return new \WP_REST_Response(['success' => $saved], $saved ? 200 : 500);
    }

    public function delete_policy(\WP_REST_Request $request): \WP_REST_Response
    {
        $id = $request->get_param('id');
        $deleted = WPG_Policy_Engine::instance()->delete_policy($id);

        return new \WP_REST_Response(['success' => $deleted], $deleted ? 200 : 404);
    }

    public function toggle_policy(\WP_REST_Request $request): \WP_REST_Response
    {
        $id = $request->get_param('id');
        $body = $request->get_json_params();
        $active = (bool)($body['active'] ?? true);

        $toggled = WPG_Policy_Engine::instance()->toggle_policy($id, $active);

        return new \WP_REST_Response(['success' => $toggled], $toggled ? 200 : 404);
    }

    // ===== Check Policy (Dry Run) =====

    public function check_policy(\WP_REST_Request $request): \WP_REST_Response
    {
        $body = $request->get_json_params();

        $context = [
            'ability' => sanitize_text_field($body['ability'] ?? ''),
            'args' => $body['args'] ?? [],
            'agent' => sanitize_text_field($body['agent'] ?? 'test'),
            'agent_type' => sanitize_text_field($body['agent_type'] ?? 'manual'),
            'user_id' => get_current_user_id(),
        ];

        $result = WPG_Policy_Engine::instance()->evaluate($context);

        return new \WP_REST_Response($result->to_array(), 200);
    }

    // ===== Settings =====

    public function get_settings(\WP_REST_Request $request): \WP_REST_Response
    {
        return new \WP_REST_Response([
            'deny_first' => (bool)get_option('wpg_deny_first', true),
            'alert_settings' => get_option('wpg_alert_settings', []),
            'agent_profiles' => get_option('wpg_agent_profiles', []),
            'version' => WPG_VERSION,
            'db_version' => get_option('wpg_db_version', '0'),
        ], 200);
    }

    public function update_settings(\WP_REST_Request $request): \WP_REST_Response
    {
        $body = $request->get_json_params();

        if (isset($body['deny_first'])) {
            update_option('wpg_deny_first', (bool)$body['deny_first']);
        }

        if (isset($body['alert_settings'])) {
            update_option('wpg_alert_settings', $body['alert_settings']);
        }

        if (isset($body['agent_profiles'])) {
            update_option('wpg_agent_profiles', $body['agent_profiles']);
        }

        return new \WP_REST_Response(['success' => true], 200);
    }

    // ===== Simulation (Full Pipeline: Evaluate + Log) =====

    public function simulate_action(\WP_REST_Request $request): \WP_REST_Response
    {
        $body = $request->get_json_params();

        $context = [
            'ability' => sanitize_text_field($body['ability'] ?? ''),
            'args' => $body['args'] ?? [],
            'agent' => sanitize_text_field($body['agent'] ?? 'test-agent'),
            'agent_type' => sanitize_text_field($body['agent_type'] ?? 'simulation'),
            'user_id' => get_current_user_id(),
            'ip' => '127.0.0.1',
            'session_id' => 'sim-' . wp_generate_uuid4(),
            'timestamp' => current_time('mysql', true),
        ];

        $result = WPG_Policy_Engine::instance()->evaluate($context);
        $log_id = WPG_Audit_Log::instance()->record($result);

        $response = $result->to_array();
        $response['logged'] = true;
        $response['log_id'] = $log_id;

        return new \WP_REST_Response($response, 200);
    }

    public function simulate_batch(\WP_REST_Request $request): \WP_REST_Response
    {
        $scenarios = [
            // ---- These SHOULD be DENIED ----
            [
                'label' => 'AI tries to create an admin user',
                'ability' => 'core/create-user',
                'args' => ['role' => 'administrator', 'user_login' => 'ai-backdoor'],
                'agent' => 'Claude Desktop',
            ],
            [
                'label' => 'AI tries to install a plugin',
                'ability' => 'core/install-plugin',
                'args' => ['slug' => 'suspicious-seo-tool'],
                'agent' => 'Cursor AI',
            ],
            [
                'label' => 'AI tries to edit a PHP file',
                'ability' => 'core/edit-file',
                'args' => ['file' => 'wp-config.php'],
                'agent' => 'ChatGPT Operator',
            ],
            [
                'label' => 'AI tries to change the site URL',
                'ability' => 'core/update-option',
                'args' => ['option_name' => 'siteurl', 'option_value' => 'https://evil.com'],
                'agent' => 'Windsurf',
            ],
            [
                'label' => 'AI tries to update a user to admin role',
                'ability' => 'core/update-user-role',
                'args' => ['user_id' => 99, 'role' => 'administrator'],
                'agent' => 'VS Code Copilot',
            ],
            [
                'label' => 'AI tries to delete the active theme',
                'ability' => 'core/delete-theme',
                'args' => ['stylesheet' => 'twentytwentyfive'],
                'agent' => 'Devin',
            ],

            // ---- These SHOULD be ALLOWED ----
            [
                'label' => 'AI creates a draft blog post',
                'ability' => 'core/create-post',
                'args' => ['post_title' => 'Weekend in Vermont', 'post_status' => 'draft'],
                'agent' => 'Claude Desktop',
            ],
            [
                'label' => 'AI reads a page (non-governed ability)',
                'ability' => 'core/get-page',
                'args' => ['page_id' => 42],
                'agent' => 'Cursor AI',
            ],
        ];

        $results = [];

        foreach ($scenarios as $scenario) {
            $context = [
                'ability' => $scenario['ability'],
                'args' => $scenario['args'],
                'agent' => $scenario['agent'],
                'agent_type' => 'simulation',
                'user_id' => get_current_user_id(),
                'ip' => '127.0.0.1',
                'session_id' => 'sim-batch-' . wp_generate_uuid4(),
                'timestamp' => current_time('mysql', true),
            ];

            $result = WPG_Policy_Engine::instance()->evaluate($context);
            $log_id = WPG_Audit_Log::instance()->record($result);

            $results[] = [
                'label' => $scenario['label'],
                'decision' => $result->decision,
                'policy' => $result->policy_name,
                'reason' => $result->reason,
                'agent' => $scenario['agent'],
                'ability' => $scenario['ability'],
                'log_id' => $log_id,
                'time_ms' => round($result->elapsed_ms, 3),
            ];
        }

        $denied = count(array_filter($results, fn($r) => $r['decision'] === 'deny'));
        $allowed = count(array_filter($results, fn($r) => $r['decision'] === 'allow'));

        return new \WP_REST_Response([
            'success' => true,
            'total' => count($results),
            'denied' => $denied,
            'allowed' => $allowed,
            'results' => $results,
        ], 200);
    }

    // ===== Permissions =====

    public function admin_permission_check(\WP_REST_Request $request): bool
    {
        return current_user_can('manage_options');
    }
}
